"use client";

import { useState, useEffect } from "react";
import SystemCard from "./SystemCard";
import DemoModal from "./DemoModal";
import Icon from "@/components/ui/AppIcon";

const allSystems = [
{
  id: "erp-empresarial",
  name: "ERP Empresarial",
  description:
  "Plataforma integrada para gestão completa de indústrias e comércios. Controle financeiro, estoque, compras, vendas e relatórios gerenciais em tempo real.",
  segment: "Indústria & Comércio",
  segmentIcon: "BuildingStorefrontIcon",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1fc55b798-1767533845086.png",
  imageAlt: "Dashboard de ERP empresarial com gráficos de desempenho financeiro e controle de estoque em monitor",
  features: [
  "Gestão financeira completa",
  "Controle de estoque em tempo real",
  "Módulo de compras e fornecedores",
  "Emissão de NF-e / NFS-e",
  "Relatórios gerenciais avançados",
  "Multi-filial e multi-empresa"],

  demoUrl: "https://example.com",
  gradientFrom: "#011363",
  gradientTo: "#2d5be3",
  accentColor: "#2d5be3"
},
{
  id: "gestao-rh",
  name: "Gestão de RH & DP",
  description:
  "Sistema completo para departamento pessoal: controle de ponto, folha de pagamento, gestão de benefícios, férias e documentos dos colaboradores.",
  segment: "Recursos Humanos",
  segmentIcon: "UserGroupIcon",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1681f78f6-1767085536563.png",
  imageAlt: "Interface de sistema de RH com perfis de colaboradores, calendário de férias e indicadores de equipe",
  features: [
  "Ponto eletrônico digital",
  "Folha de pagamento automatizada",
  "Gestão de férias e afastamentos",
  "Controle de benefícios",
  "Integração eSocial",
  "Portal do colaborador"],

  demoUrl: "https://example.com",
  gradientFrom: "#7c3aed",
  gradientTo: "#2d5be3",
  accentColor: "#7c3aed"
},
{
  id: "hub-marketplace",
  name: "Hub de Marketplace",
  description:
  "Centralize todas as suas vendas online em um único painel. Sincronize estoque, pedidos e precificação entre Shopee, Mercado Livre, Amazon e loja própria.",
  segment: "E-commerce",
  segmentIcon: "ShoppingCartIcon",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_107d6d6ef-1769399422682.png",
  imageAlt: "Painel de controle de marketplace com gráficos de vendas por canal e status de pedidos em andamento",
  features: [
  "Multi-canal integrado",
  "Estoque sincronizado",
  "Gestão de pedidos unificada",
  "Precificação dinâmica",
  "Emissão automática de NF-e",
  "Relatórios por canal"],

  demoUrl: "https://example.com",
  gradientFrom: "#ea580c",
  gradientTo: "#dc2626",
  accentColor: "#ea580c"
},
{
  id: "gestao-financeira",
  name: "Gestão Financeira",
  description:
  "Controle total do fluxo de caixa, contas a pagar e receber, conciliação bancária automatizada e DRE em tempo real para tomada de decisão estratégica.",
  segment: "Financeiro",
  segmentIcon: "BanknotesIcon",
  image: "https://images.unsplash.com/photo-1724833256463-26b199dc1b69",
  imageAlt: "Tela de gestão financeira com gráfico de fluxo de caixa, saldo de contas e indicadores de DRE",
  features: [
  "Fluxo de caixa em tempo real",
  "Contas a pagar e receber",
  "Conciliação bancária automática",
  "DRE e balanço patrimonial",
  "Integração com bancos via API",
  "Controle de centros de custo"],

  demoUrl: "https://example.com",
  gradientFrom: "#059669",
  gradientTo: "#0d9488",
  accentColor: "#059669"
},
{
  id: "crm-vendas",
  name: "CRM & Força de Vendas",
  description:
  "Gerencie todo o pipeline comercial, oportunidades, contatos e atividades da equipe de vendas. Funil visual com automações e métricas de conversão.",
  segment: "Comercial",
  segmentIcon: "PresentationChartLineIcon",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_133a41f4a-1770478458756.png",
  imageAlt: "CRM com pipeline de vendas visual em kanban, cards de oportunidades e métricas de conversão por etapa",
  features: [
  "Pipeline kanban customizável",
  "Gestão de leads e contatos",
  "Automação de follow-ups",
  "Propostas e contratos digitais",
  "Relatórios de desempenho",
  "Integração com WhatsApp"],

  demoUrl: "https://example.com",
  gradientFrom: "#0284c7",
  gradientTo: "#0369a1",
  accentColor: "#0284c7"
},
{
  id: "gestao-clinica",
  name: "Gestão de Clínica",
  description:
  "Sistema completo para clínicas médicas e odontológicas: agendamento online, prontuário eletrônico, controle financeiro e gestão de convênios.",
  segment: "Saúde",
  segmentIcon: "HeartIcon",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1a07fec34-1764641716124.png",
  imageAlt: "Interface de gestão de clínica com agenda médica, prontuário eletrônico e painel de atendimentos do dia",
  features: [
  "Agendamento online integrado",
  "Prontuário eletrônico (PEP)",
  "Controle de convênios",
  "Faturamento TISS automatizado",
  "Gestão de estoque de insumos",
  "Relatórios de produtividade"],

  demoUrl: "https://example.com",
  gradientFrom: "#be185d",
  gradientTo: "#9d174d",
  accentColor: "#be185d"
}];


const segments = ["Todos", "Indústria & Comércio", "Recursos Humanos", "E-commerce", "Financeiro", "Comercial", "Saúde"];

export default function SystemsGrid() {
  const [activeSegment, setActiveSegment] = useState("Todos");
  const [modalData, setModalData] = useState<{name: string;demoUrl: string;segment: string;} | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);

  const filtered =
  activeSegment === "Todos" ?
  allSystems :
  allSystems.filter((s) => s.segment === activeSegment);

  const handleDemoClick = (system: {name: string;demoUrl: string;segment: string;}) => {
    setModalData(system);
    setIsModalOpen(true);
  };

  const handleCloseModal = () => {
    setIsModalOpen(false);
    setTimeout(() => setModalData(null), 300);
  };

  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) entry.target.classList.add("visible");
        });
      },
      { threshold: 0.1 }
    );
    document.querySelectorAll(".reveal").forEach((el) => observer.observe(el));
    return () => observer.disconnect();
  }, [filtered]);

  return (
    <>
      {/* Filter Tabs */}
      <div className="flex flex-wrap gap-2 mb-10 reveal">
        {segments.map((seg) =>
        <button
          key={seg}
          onClick={() => setActiveSegment(seg)}
          className={`px-4 py-2 rounded-full text-sm font-semibold font-heading transition-all duration-200 ${
          activeSegment === seg ?
          "bg-primary text-white shadow-card" :
          "bg-white text-gray-500 border border-gray-200 hover:border-primary/30 hover:text-primary hover:bg-primary-50"}`
          }>
          
            {seg}
          </button>
        )}
      </div>

      {/* Count */}
      <p className="text-sm text-gray-400 mb-8 reveal delay-100">
        {filtered.length} sistema{filtered.length !== 1 ? "s" : ""} encontrado{filtered.length !== 1 ? "s" : ""}
        {activeSegment !== "Todos" && ` em ${activeSegment}`}
      </p>

      {/* Grid */}
      <div className="grid md:grid-cols-2 xl:grid-cols-3 gap-8">
        {filtered.map((system, i) =>
        <SystemCard
          key={system.id}
          system={system}
          onDemoClick={handleDemoClick}
          animDelay={`delay-${Math.min((i + 1) * 100, 600)}`} />

        )}
      </div>

      {/* Empty state */}
      {filtered.length === 0 &&
      <div className="text-center py-20">
          <div className="w-16 h-16 rounded-2xl bg-primary-50 flex items-center justify-center mx-auto mb-4">
            <Icon name="MagnifyingGlassIcon" size={28} className="text-primary" />
          </div>
          <p className="text-gray-500 font-medium">Nenhum sistema nessa categoria ainda.</p>
        </div>
      }

      {/* Demo Modal */}
      <DemoModal
        isOpen={isModalOpen}
        onClose={handleCloseModal}
        system={modalData} />
      
    </>);

}