"use client";

import { useEffect } from "react";
import Link from "next/link";
import Icon from "@/components/ui/AppIcon";
import AppImage from "@/components/ui/AppImage";

const previewSystems = [
{
  id: "erp",
  name: "ERP Empresarial",
  description: "Gestão integrada de financeiro, estoque, vendas e compras em uma única plataforma.",
  segment: "Indústria & Comércio",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1fc55b798-1767533845086.png",
  alt: "Dashboard de ERP empresarial com gráficos de vendas e indicadores financeiros em tela de computador",
  tags: ["Financeiro", "Estoque", "Compras"],
  color: "from-blue-600 to-primary"
},
{
  id: "rh",
  name: "Gestão de RH",
  description: "Controle completo de funcionários, ponto eletrônico, folha de pagamento e benefícios.",
  segment: "Recursos Humanos",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1d73d3f78-1768646246752.png",
  alt: "Interface de sistema de gestão de recursos humanos com lista de funcionários e métricas de equipe",
  tags: ["Ponto Digital", "Folha", "Benefícios"],
  color: "from-purple-600 to-accent"
},
{
  id: "marketplace",
  name: "Hub de Marketplace",
  description: "Centralize vendas de múltiplos canais: Shopee, Mercado Livre, Amazon e loja própria.",
  segment: "E-commerce",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_107d6d6ef-1769399422682.png",
  alt: "Painel de controle de marketplace mostrando vendas em múltiplos canais e status de pedidos",
  tags: ["Multi-canal", "Estoque", "Pedidos"],
  color: "from-orange-500 to-red-500"
}];


export default function SystemsPreviewSection() {
  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) entry.target.classList.add("visible");
        });
      },
      { threshold: 0.1 }
    );
    document.querySelectorAll(".reveal, .reveal-scale")?.forEach((el) => observer?.observe(el));
    return () => observer?.disconnect();
  }, []);

  return (
    <section className="py-24 bg-white">
      <div className="max-w-7xl mx-auto px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <span className="section-label mb-4 reveal">
            <Icon name="RectangleGroupIcon" size={12} />
            Catálogo de Soluções
          </span>
          <h2 className="font-heading font-extrabold text-4xl lg:text-5xl text-primary-dark mt-4 mb-5 reveal delay-100">
            Sistemas que já
            <span className="gradient-text"> transformaram empresas</span>
          </h2>
          <p className="text-gray-500 max-w-xl mx-auto leading-relaxed reveal delay-200">
            Veja alguns dos sistemas que desenvolvemos para nossos clientes.
            Cada um foi construído do zero para atender necessidades específicas.
          </p>
        </div>

        {/* Cards */}
        <div className="grid lg:grid-cols-3 gap-8 mb-12">
          {previewSystems?.map((system, i) =>
          <div
            key={system?.id}
            className={`system-card reveal delay-${(i + 1) * 100}`}>
            
              {/* Image */}
              <div className="relative overflow-hidden h-48">
                <AppImage
                src={system?.image}
                alt={system?.alt}
                fill
                className="object-cover transition-transform duration-700 group-hover:scale-110" />
              
                <div className={`absolute inset-0 bg-gradient-to-t ${system?.color} opacity-60`} />
                <div className="absolute bottom-4 left-4">
                  <span className="text-white font-heading font-bold text-lg">{system?.name}</span>
                </div>
              </div>

              {/* Content */}
              <div className="p-6">
                <span className="segment-badge mb-3">
                  <Icon name="TagIcon" size={10} />
                  {system?.segment}
                </span>
                <p className="text-sm text-gray-500 leading-relaxed mb-4">{system?.description}</p>
                <div className="flex flex-wrap gap-1.5 mb-5">
                  {system?.tags?.map((tag) =>
                <span key={tag} className="text-xs text-primary bg-primary-50 rounded-full px-2.5 py-1 font-medium">
                      {tag}
                    </span>
                )}
                </div>
                <Link
                href="/systems"
                className="flex items-center gap-2 text-sm font-bold text-primary hover:text-accent transition-colors font-heading">
                
                  Ver detalhes
                  <Icon name="ArrowRightIcon" size={14} />
                </Link>
              </div>
            </div>
          )}
        </div>

        {/* CTA */}
        <div className="text-center reveal">
          <Link href="/systems" className="btn-primary text-base px-10 py-4">
            <Icon name="RectangleGroupIcon" size={18} />
            Ver Todos os Sistemas
          </Link>
        </div>
      </div>
    </section>);

}