"use client";

import { useState, useEffect } from "react";
import Link from "next/link";
import { usePathname } from "next/navigation";
import AppLogo from "@/components/ui/AppLogo";
import Icon from "@/components/ui/AppIcon";

const navLinks = [
  { label: "Início", href: "/homepage" },
  { label: "Sistemas", href: "/systems" },
  { label: "Preços", href: "/pricing" },
];

export default function Header() {
  const pathname = usePathname();
  const [isScrolled, setIsScrolled] = useState(false);
  const [mobileOpen, setMobileOpen] = useState(false);

  useEffect(() => {
    const handleScroll = () => setIsScrolled(window.scrollY > 20);
    window.addEventListener("scroll", handleScroll, { passive: true });
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  useEffect(() => {
    setMobileOpen(false);
  }, [pathname]);

  useEffect(() => {
    document.body.style.overflow = mobileOpen ? "hidden" : "";
    return () => { document.body.style.overflow = ""; };
  }, [mobileOpen]);

  return (
    <>
      <header
        className={`fixed top-0 left-0 right-0 z-50 transition-all duration-500 ${
          isScrolled
            ? "bg-white/95 backdrop-blur-xl shadow-[0_2px_20px_rgba(1,19,99,0.08)] border-b border-[#E2E5F0]"
            : "bg-transparent"
        }`}
      >
        <div className="max-w-7xl mx-auto px-6 lg:px-8">
          <div className="flex items-center justify-between h-20">
            {/* Logo */}
            <Link href="/homepage" className="flex items-center gap-3 group">
              <AppLogo
                size={40}
                text="ADVOPACK"
                iconName="CpuChipIcon"
                className="transition-transform duration-300 group-hover:scale-105"
              />
            </Link>

            {/* Desktop Nav */}
            <nav className="hidden md:flex items-center gap-8">
              {navLinks?.map((link) => (
                <Link
                  key={link?.href}
                  href={link?.href}
                  className={`nav-link ${pathname === link?.href ? "active" : ""}`}
                >
                  {link?.label}
                </Link>
              ))}
            </nav>

            {/* CTA */}
            <div className="hidden md:flex items-center gap-4">
              <Link
                href="/homepage#contact"
                className="btn-primary text-sm py-2.5 px-5"
              >
                <span>Solicitar Demo</span>
                <Icon name="ArrowRightIcon" size={16} />
              </Link>
            </div>

            {/* Mobile Toggle */}
            <button
              onClick={() => setMobileOpen(!mobileOpen)}
              className="md:hidden p-2 rounded-xl text-[#011363] hover:bg-[#EEF0FA] transition-colors"
              aria-label="Toggle menu"
            >
              <Icon name={mobileOpen ? "XMarkIcon" : "Bars3Icon"} size={24} />
            </button>
          </div>
        </div>
      </header>
      {/* Mobile Menu */}
      <div
        className={`fixed inset-0 z-40 md:hidden transition-all duration-400 ${
          mobileOpen ? "opacity-100 pointer-events-auto" : "opacity-0 pointer-events-none"
        }`}
      >
        <div
          className="absolute inset-0 bg-[#010D45]/60 backdrop-blur-sm"
          onClick={() => setMobileOpen(false)}
        />
        <div
          className={`absolute top-0 right-0 bottom-0 w-72 bg-white shadow-2xl transition-transform duration-400 ${
            mobileOpen ? "translate-x-0" : "translate-x-full"
          }`}
          style={{ transitionTimingFunction: "cubic-bezier(0.25, 1, 0.5, 1)" }}
        >
          <div className="p-6 border-b border-[#E2E5F0]">
            <div className="flex items-center justify-between">
              <AppLogo size={36} text="ADVOPACK" iconName="CpuChipIcon" />
              <button
                onClick={() => setMobileOpen(false)}
                className="p-2 rounded-xl text-[#011363] hover:bg-[#EEF0FA] transition-colors"
              >
                <Icon name="XMarkIcon" size={20} />
              </button>
            </div>
          </div>
          <nav className="p-6 flex flex-col gap-2">
            {navLinks?.map((link) => (
              <Link
                key={link?.href}
                href={link?.href}
                className={`flex items-center gap-3 px-4 py-3 rounded-xl font-heading font-500 text-sm transition-all duration-200 ${
                  pathname === link?.href
                    ? "bg-[#EEF0FA] text-[#011363] font-semibold"
                    : "text-[#0D1340] hover:bg-[#F8F9FF] hover:text-[#011363]"
                }`}
              >
                {link?.label}
              </Link>
            ))}
            <div className="mt-4 pt-4 border-t border-[#E2E5F0]">
              <Link
                href="/homepage#contact"
                className="btn-primary w-full justify-center text-sm"
              >
                <span>Solicitar Demo</span>
                <Icon name="ArrowRightIcon" size={16} />
              </Link>
            </div>
          </nav>
        </div>
      </div>
    </>
  );
}