"use client";

import { useState, useEffect, useRef } from "react";
import Link from "next/link";
import Icon from "@/components/ui/AppIcon";
import AppImage from "@/components/ui/AppImage";

interface System {
  id: string;
  name: string;
  description: string;
  segment: string;
  segmentColor: string;
  image: string;
  imageAlt: string;
  features: string[];
  demoUrl: string;
  badge?: string;
}

const systems: System[] = [
{
  id: "erp-varejo",
  name: "ERP Varejo Pro",
  description:
  "Sistema completo de gestão para varejistas, integrando PDV, estoque, financeiro e CRM em uma única plataforma.",
  segment: "Varejo",
  segmentColor: "#2563EB",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1ad014123-1766866232190.png",
  imageAlt: "Dashboard do ERP Varejo Pro com gráficos de vendas e controle de estoque em tela de computador",
  features: [
  "PDV integrado",
  "Controle de estoque",
  "Gestão financeira",
  "CRM básico",
  "Relatórios de vendas",
  "Nota fiscal eletrônica"],

  demoUrl: "https://example.com",
  badge: "Mais popular"
},
{
  id: "gestao-rh",
  name: "RH & Pessoas",
  description:
  "Plataforma de gestão de recursos humanos com controle de ponto, folha de pagamento, férias e avaliações de desempenho.",
  segment: "RH",
  segmentColor: "#059669",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1887654c0-1767455023704.png",
  imageAlt: "Interface do sistema de RH mostrando painel de funcionários, controle de ponto e métricas de equipe",
  features: [
  "Controle de ponto",
  "Folha de pagamento",
  "Gestão de férias",
  "Avaliação de desempenho",
  "Organograma",
  "Portal do colaborador"],

  demoUrl: "https://example.com"
},
{
  id: "financeiro",
  name: "FinanceDesk",
  description:
  "Sistema financeiro completo para gestão de contas, fluxo de caixa, conciliação bancária e relatórios gerenciais.",
  segment: "Financeiro",
  segmentColor: "#D97706",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1cda1434f-1767355564993.png",
  imageAlt: "Tela do FinanceDesk exibindo gráficos de fluxo de caixa, contas a pagar e relatórios financeiros",
  features: [
  "Contas a pagar/receber",
  "Fluxo de caixa",
  "Conciliação bancária",
  "DRE automatizado",
  "Centros de custo",
  "Open Banking"],

  demoUrl: "https://example.com"
},
{
  id: "logistica",
  name: "LogiTrack",
  description:
  "Gestão logística inteligente com rastreamento de entregas, controle de frota, roteirização e integração com marketplaces.",
  segment: "Logística",
  segmentColor: "#7C3AED",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_161219931-1764982534143.png",
  imageAlt: "Dashboard do LogiTrack com mapa de rastreamento de frotas, status de entregas e rotas otimizadas",
  features: [
  "Rastreamento em tempo real",
  "Roteirização inteligente",
  "Controle de frota",
  "Integração marketplaces",
  "Gestão de devoluções",
  "App motorista"],

  demoUrl: "https://example.com",
  badge: "Novo"
},
{
  id: "clinica",
  name: "MedSystem",
  description:
  "Software médico completo para clínicas e consultórios, com agendamento online, prontuário eletrônico e faturamento.",
  segment: "Saúde",
  segmentColor: "#DC2626",
  image: "https://img.rocket.new/generatedImages/rocket_gen_img_1693a27a7-1764643565140.png",
  imageAlt: "Interface do MedSystem mostrando agenda de consultas, prontuário eletrônico e painel de atendimentos",
  features: [
  "Agendamento online",
  "Prontuário eletrônico",
  "Faturamento TISS",
  "Controle de estoque",
  "Telemedicina",
  "Portal do paciente"],

  demoUrl: "https://example.com"
},
{
  id: "industria",
  name: "IndusPlan",
  description:
  "Sistema MES para indústria com controle de produção, PCP, manutenção de equipamentos e rastreabilidade.",
  segment: "Indústria",
  segmentColor: "#0A1F8F",
  image: "https://images.unsplash.com/photo-1557053198-a8c0fa8b6c12",
  imageAlt: "Dashboard do IndusPlan exibindo linha de produção, ordens de fabricação e indicadores de eficiência industrial",
  features: [
  "Controle de produção",
  "PCP automatizado",
  "Manutenção preventiva",
  "Rastreabilidade",
  "OEE em tempo real",
  "Integração ERP"],

  demoUrl: "https://example.com"
}];


interface ModalState {
  open: boolean;
  system: System | null;
}

export default function SystemsInteractive() {
  const [modal, setModal] = useState<ModalState>({ open: false, system: null });
  const [filter, setFilter] = useState<string>("Todos");
  const sectionRef = useRef<HTMLDivElement>(null);

  const segments = ["Todos", ...Array.from(new Set(systems.map((s) => s.segment)))];

  const filtered =
  filter === "Todos" ? systems : systems.filter((s) => s.segment === filter);

  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            entry.target.querySelectorAll(".reveal-hidden").forEach((el) => {
              el.classList.remove("reveal-hidden");
            });
          }
        });
      },
      { threshold: 0.05 }
    );
    if (sectionRef.current) observer.observe(sectionRef.current);
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    if (modal.open) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "";
    }
    return () => {document.body.style.overflow = "";};
  }, [modal.open]);

  const openModal = (system: System) => {
    setModal({ open: true, system });
  };

  const closeModal = () => {
    setModal({ open: false, system: null });
  };

  return (
    <>
      {/* Page Header */}
      <section
        className="pt-32 pb-16 relative overflow-hidden"
        style={{
          background: "linear-gradient(135deg, #F8F9FF 0%, #EEF0FA 60%, #E0E4F5 100%)"
        }}>
        
        <div
          className="absolute inset-0 grid-bg opacity-50" />
        
        <div className="relative z-10 max-w-7xl mx-auto px-6 lg:px-8">
          <div className="max-w-3xl">
            <div className="section-label inline-flex mb-4 animate-fade-in-up">
              <Icon name="Squares2X2Icon" size={14} />
              Catálogo de Soluções
            </div>
            <h1
              className="font-heading text-5xl lg:text-6xl font-extrabold text-[#0D1340] tracking-tight leading-tight mb-6 animate-fade-in-up"
              style={{ animationDelay: "0.1s" }}>
              
              Sistemas{" "}
              <span className="gradient-text">desenvolvidos</span>{" "}
              e entregues
            </h1>
            <p
              className="text-[#6B7280] text-lg leading-relaxed max-w-2xl animate-fade-in-up"
              style={{ animationDelay: "0.2s" }}>
              
              Conheça exemplos reais de sistemas que a ADVOPACK construiu para
              diferentes segmentos. Cada solução foi desenvolvida sob medida para
              a operação específica de cada cliente.
            </p>
          </div>
        </div>
        <div className="absolute bottom-0 left-0 right-0">
          <svg viewBox="0 0 1440 60" fill="none" className="w-full">
            <path d="M0 60L1440 60L1440 20C1200 60 960 0 720 15C480 30 240 60 0 20Z" fill="white" />
          </svg>
        </div>
      </section>

      {/* Systems Catalog */}
      <section className="py-16 bg-white" ref={sectionRef}>
        <div className="max-w-7xl mx-auto px-6 lg:px-8">
          {/* Filter Tabs */}
          <div className="flex flex-wrap gap-2 mb-10 reveal reveal-hidden">
            {segments.map((seg) =>
            <button
              key={seg}
              onClick={() => setFilter(seg)}
              className={`px-4 py-2 rounded-full text-sm font-semibold font-heading transition-all duration-200 ${
              filter === seg ?
              "bg-[#011363] text-white shadow-md" :
              "bg-[#EEF0FA] text-[#6B7280] hover:bg-[#E0E4F5] hover:text-[#011363]"}`
              }>
              
                {seg}
              </button>
            )}
          </div>

          {/* Grid */}
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filtered.map((system, i) =>
            <div
              key={system.id}
              className={`system-card group reveal reveal-hidden reveal-delay-${i % 3 + 1}`}>
              
                {/* Image */}
                <div className="relative h-44 overflow-hidden">
                  <AppImage
                  src={system.image}
                  alt={system.imageAlt}
                  width={600}
                  height={176}
                  className="w-full h-full object-cover transition-transform duration-700 group-hover:scale-105" />
                
                  <div className="absolute inset-0 bg-gradient-to-t from-[#011363]/40 to-transparent" />

                  {/* Segment Tag */}
                  <div className="absolute top-3 left-3">
                    <span
                    className="tag text-white font-bold"
                    style={{ background: system.segmentColor }}>
                    
                      {system.segment}
                    </span>
                  </div>

                  {/* Badge */}
                  {system.badge &&
                <div className="absolute top-3 right-3">
                      <span className="tag bg-amber-400 text-amber-900 font-bold">
                        {system.badge}
                      </span>
                    </div>
                }
                </div>

                {/* Content */}
                <div className="p-6">
                  <h3 className="font-heading font-bold text-xl text-[#0D1340] mb-2">
                    {system.name}
                  </h3>
                  <p className="text-sm text-[#6B7280] leading-relaxed mb-5">
                    {system.description}
                  </p>

                  {/* Features */}
                  <div className="mb-6">
                    <div className="text-xs font-bold uppercase tracking-wider text-[#9ea3ba] mb-3">
                      Funcionalidades
                    </div>
                    <div className="grid grid-cols-2 gap-1.5">
                      {system.features.map((feat) =>
                    <div
                      key={feat}
                      className="flex items-center gap-1.5 text-xs text-[#6B7280]">
                      
                          <Icon
                        name="CheckCircleIcon"
                        size={12}
                        className="text-[#011363] flex-shrink-0" />
                      
                          {feat}
                        </div>
                    )}
                    </div>
                  </div>

                  {/* Actions */}
                  <div className="flex gap-3 pt-4 border-t border-[#E2E5F0]">
                    <button
                    onClick={() => openModal(system)}
                    className="btn-primary flex-1 justify-center text-sm py-2.5">
                    
                      <Icon name="PlayIcon" size={15} />
                      <span>Ver Demo</span>
                    </button>
                    <Link
                    href="/homepage#contact"
                    className="flex items-center justify-center gap-1.5 px-4 py-2.5 rounded-xl border border-[#E2E5F0] text-sm font-semibold text-[#011363] hover:bg-[#EEF0FA] hover:border-[#011363]/20 transition-all duration-200">
                    
                      <Icon name="EnvelopeIcon" size={15} />
                      Solicitar
                    </Link>
                  </div>
                </div>
              </div>
            )}
          </div>

          {/* Bottom CTA */}
          <div className="mt-16 text-center reveal reveal-hidden">
            <div className="bg-[#EEF0FA] rounded-3xl p-10 max-w-2xl mx-auto">
              <div className="w-14 h-14 rounded-2xl bg-primary-gradient flex items-center justify-center mx-auto mb-5">
                <Icon name="PlusCircleIcon" size={28} className="text-white" />
              </div>
              <h3 className="font-heading font-bold text-2xl text-[#0D1340] mb-3">
                Não encontrou o que precisa?
              </h3>
              <p className="text-[#6B7280] text-sm leading-relaxed mb-6">
                Desenvolvemos qualquer sistema sob medida para sua operação.
                Fale conosco e receba uma proposta personalizada.
              </p>
              <Link href="/homepage#contact" className="btn-primary text-sm">
                <Icon name="ChatBubbleLeftIcon" size={16} />
                <span>Solicitar Sistema Personalizado</span>
              </Link>
            </div>
          </div>
        </div>
      </section>

      {/* Demo Modal */}
      {modal.open && modal.system &&
      <div className="modal-overlay animate-modal-backdrop" onClick={closeModal}>
          <div
          className="modal-container"
          onClick={(e) => e.stopPropagation()}>
          
            {/* Modal Header */}
            <div className="modal-header">
              <div className="flex items-center gap-3">
                <div className="w-8 h-8 rounded-lg bg-white/10 flex items-center justify-center">
                  <Icon name="ComputerDesktopIcon" size={18} className="text-white" />
                </div>
                <div>
                  <div className="text-white font-heading font-bold text-sm">
                    {modal.system.name}
                  </div>
                  <div className="text-white/60 text-xs">
                    Demonstração interativa
                  </div>
                </div>
              </div>

              <div className="flex items-center gap-2">
                {/* URL Bar */}
                <div className="hidden sm:flex items-center gap-2 bg-white/10 rounded-lg px-3 py-1.5 text-xs text-white/70 max-w-xs">
                  <Icon name="LockClosedIcon" size={12} className="text-green-400 flex-shrink-0" />
                  <span className="truncate">{modal.system.demoUrl}</span>
                </div>

                {/* Open in new tab */}
                <a
                href={modal.system.demoUrl}
                target="_blank"
                rel="noopener noreferrer"
                className="flex items-center gap-1.5 bg-white/10 hover:bg-white/20 transition-colors text-white text-xs font-semibold px-3 py-1.5 rounded-lg">
                
                  <Icon name="ArrowTopRightOnSquareIcon" size={14} />
                  <span className="hidden sm:inline">Abrir em nova aba</span>
                </a>

                {/* Close */}
                <button
                onClick={closeModal}
                className="w-8 h-8 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center text-white transition-colors"
                aria-label="Fechar demo">
                
                  <Icon name="XMarkIcon" size={18} />
                </button>
              </div>
            </div>

            {/* iframe */}
            <div className="flex-1 relative bg-[#F0F0F0]">
              {/* Loading state overlay */}
              <div className="absolute inset-0 flex flex-col items-center justify-center bg-[#F8F9FF] z-10 pointer-events-none">
                <div className="w-16 h-16 rounded-2xl bg-primary-gradient flex items-center justify-center mb-4 animate-pulse-glow">
                  <Icon name="ComputerDesktopIcon" size={32} className="text-white" />
                </div>
                <div className="font-heading font-bold text-[#0D1340] text-lg mb-2">
                  {modal.system.name}
                </div>
                <div className="text-[#9ea3ba] text-sm">
                  Carregando demonstração...
                </div>
              </div>
              <iframe
              src={modal.system.demoUrl}
              title={`Demo: ${modal.system.name}`}
              className="modal-iframe relative z-20"
              sandbox="allow-same-origin allow-scripts allow-forms allow-popups"
              onLoad={(e) => {
                const loadingEl = (e.target as HTMLIFrameElement).
                parentElement?.
                querySelector(".z-10") as HTMLElement;
                if (loadingEl) loadingEl.style.display = "none";
              }} />
            
            </div>
          </div>
        </div>
      }
    </>);

}