"use client";

import { useState, useEffect, useRef } from "react";
import Link from "next/link";
import Icon from "@/components/ui/AppIcon";

const pricingPlans = [
  {
    id: "sob-medida",
    name: "Projeto Sob Medida",
    tagline: "Para empresas que precisam de um sistema único",
    price: "Sob orçamento",
    priceNote: "Conforme escopo e complexidade",
    highlight: false,
    badge: null,
    description:
      "Desenvolvemos do zero um sistema 100% alinhado à sua operação. Da análise de requisitos ao go-live, com toda a documentação e treinamento inclusos.",
    features: [
      { text: "Análise completa da operação", included: true },
      { text: "Arquitetura personalizada", included: true },
      { text: "Desenvolvimento ágil (sprints)", included: true },
      { text: "Testes e homologação", included: true },
      { text: "Treinamento da equipe", included: true },
      { text: "Documentação técnica", included: true },
      { text: "Suporte pós-entrega (3 meses)", included: true },
      { text: "Manutenção evolutiva", included: false },
    ],
    cta: "Solicitar Análise",
    ctaHref: "/homepage#contact",
    icon: "WrenchScrewdriverIcon" as const,
    color: "#011363",
    bg: "#EEF0FA",
  },
  {
    id: "integracoes",
    name: "Integrações Específicas",
    tagline: "Para quem precisa conectar sistemas existentes",
    price: "A partir de R$ 2.500",
    priceNote: "Por integração entregue",
    highlight: true,
    badge: "Mais contratado",
    description:
      "Conectamos seu sistema atual a qualquer API ou plataforma externa — ERP fiscal, marketplace, gateway de pagamento, CRM, ERP e muito mais.",
    features: [
      { text: "Mapeamento da integração", included: true },
      { text: "Desenvolvimento da API", included: true },
      { text: "Autenticação OAuth/JWT", included: true },
      { text: "Tratamento de erros e logs", included: true },
      { text: "Monitoramento em tempo real", included: true },
      { text: "Documentação Swagger", included: true },
      { text: "Suporte técnico dedicado", included: true },
      { text: "SLA de disponibilidade 99,9%", included: true },
    ],
    cta: "Solicitar Proposta",
    ctaHref: "/homepage#contact",
    icon: "ArrowsPointingInIcon" as const,
    color: "#2563EB",
    bg: "#EFF6FF",
  },
  {
    id: "licenciamento",
    name: "Licenciamento + Suporte",
    tagline: "Para sistemas já implantados em operação",
    price: "A partir de R$ 890",
    priceNote: "Por mês (contrato anual)",
    highlight: false,
    badge: null,
    description:
      "Mantenha seu sistema sempre atualizado, seguro e com suporte especializado. Inclui atualizações, novas funcionalidades e atendimento prioritário.",
    features: [
      { text: "Hospedagem cloud gerenciada", included: true },
      { text: "Backups automáticos diários", included: true },
      { text: "Atualizações de segurança", included: true },
      { text: "Suporte via ticket e WhatsApp", included: true },
      { text: "Monitoramento 24/7", included: true },
      { text: "Melhorias evolutivas mensais", included: true },
      { text: "Relatório mensal de performance", included: true },
      { text: "SLA de resposta em 4h", included: true },
    ],
    cta: "Contratar Suporte",
    ctaHref: "/homepage#contact",
    icon: "ShieldCheckIcon" as const,
    color: "#059669",
    bg: "#ECFDF5",
  },
];

const faqs = [
  {
    q: "Como funciona o processo de orçamento?",
    a: "Após entrar em contato, realizamos uma reunião de descoberta para entender sua operação em profundidade. Com base nisso, elaboramos uma proposta detalhada com escopo, prazo e investimento estimado — sem compromisso da sua parte.",
  },
  {
    q: "Quanto tempo leva para entregar um sistema?",
    a: "Depende da complexidade. Sistemas simples podem ser entregues em 4 a 8 semanas. Sistemas mais robustos, com múltiplos módulos e integrações, levam entre 3 a 6 meses. Trabalhamos com entregas parciais em sprints.",
  },
  {
    q: "O sistema pode ser modificado após a entrega?",
    a: "Sim, absolutamente. Isso é uma das nossas diferenciais. Seu sistema pode evoluir conforme sua empresa cresce. Melhorias e novos módulos são tratados como projetos evolutivos com orçamento específico.",
  },
  {
    q: "Vocês oferecem suporte após a entrega?",
    a: "Sim. Todos os projetos incluem um período de suporte pós-entrega. Após esse período, você pode contratar nosso plano de Licenciamento + Suporte para manter o sistema sempre atualizado e com atendimento prioritário.",
  },
  {
    q: "Posso ter o código-fonte do meu sistema?",
    a: "Sim. Entregamos o código-fonte completo ao cliente ao final do projeto. Você terá total propriedade sobre o software desenvolvido para você.",
  },
];

export default function PricingInteractive() {
  const [openFaq, setOpenFaq] = useState<number | null>(null);
  const sectionRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            entry.target.querySelectorAll(".reveal-hidden").forEach((el) => {
              el.classList.remove("reveal-hidden");
            });
          }
        });
      },
      { threshold: 0.05 }
    );
    if (sectionRef.current) observer.observe(sectionRef.current);
    return () => observer.disconnect();
  }, []);

  return (
    <div ref={sectionRef}>
      {/* Page Header */}
      <section
        className="pt-32 pb-16 relative overflow-hidden"
        style={{
          background: "linear-gradient(135deg, #F8F9FF 0%, #EEF0FA 60%, #E0E4F5 100%)",
        }}
      >
        <div className="absolute inset-0 grid-bg opacity-50" />
        <div className="relative z-10 max-w-7xl mx-auto px-6 lg:px-8 text-center">
          <div className="section-label inline-flex mb-4 animate-fade-in-up">
            <Icon name="CurrencyDollarIcon" size={14} />
            Modelos de Contratação
          </div>
          <h1
            className="font-heading text-5xl lg:text-6xl font-extrabold text-[#0D1340] tracking-tight leading-tight mb-6 max-w-3xl mx-auto animate-fade-in-up"
            style={{ animationDelay: "0.1s" }}
          >
            Sem sistemas{" "}
            <span className="gradient-text">engessados.</span>{" "}
            Sem planos prontos.
          </h1>
          <p
            className="text-[#6B7280] text-lg leading-relaxed max-w-2xl mx-auto animate-fade-in-up"
            style={{ animationDelay: "0.2s" }}
          >
            Na ADVOPACK, cada projeto tem seu próprio orçamento, baseado na
            complexidade e nas necessidades reais da sua operação. Conheça nossos
            modelos de contratação.
          </p>
        </div>
        <div className="absolute bottom-0 left-0 right-0">
          <svg viewBox="0 0 1440 60" fill="none" className="w-full">
            <path d="M0 60L1440 60L1440 20C1200 60 960 0 720 15C480 30 240 60 0 20Z" fill="white" />
          </svg>
        </div>
      </section>

      {/* Philosophy Banner */}
      <section className="py-12 bg-white">
        <div className="max-w-7xl mx-auto px-6 lg:px-8">
          <div
            className="rounded-3xl p-8 lg:p-10 reveal reveal-hidden"
            style={{
              background: "linear-gradient(135deg, #010D45 0%, #011363 50%, #0A1F8F 100%)",
            }}
          >
            <div className="grid md:grid-cols-3 gap-6 lg:gap-10">
              {[
                {
                  icon: "NoSymbolIcon" as const,
                  title: "Sem mensalidade surpresa",
                  desc: "Você sabe exatamente o que está pagando. Sem cobranças ocultas ou funcionalidades bloqueadas.",
                },
                {
                  icon: "PuzzlePieceIcon" as const,
                  title: "100% do que você precisa",
                  desc: "Nenhum módulo que você não vai usar. Nenhuma limitação artificial. Só o que sua operação exige.",
                },
                {
                  icon: "ArrowTrendingUpIcon" as const,
                  title: "Investimento que escala",
                  desc: "Seu sistema cresce com você. Novas funcionalidades são adicionadas conforme a necessidade.",
                },
              ].map((item, i) => (
                <div key={item.title} className={`flex items-start gap-4 reveal reveal-hidden reveal-delay-${i + 1}`}>
                  <div className="w-12 h-12 rounded-2xl bg-white/10 flex items-center justify-center flex-shrink-0">
                    <Icon name={item.icon} size={22} className="text-white" />
                  </div>
                  <div>
                    <div className="font-heading font-bold text-white text-base mb-1">
                      {item.title}
                    </div>
                    <div className="text-white/60 text-sm leading-relaxed">
                      {item.desc}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Cards */}
      <section className="py-16 bg-white">
        <div className="max-w-7xl mx-auto px-6 lg:px-8">
          <div className="grid md:grid-cols-3 gap-6 lg:gap-8">
            {pricingPlans.map((plan, i) => (
              <div
                key={plan.id}
                className={`pricing-card ${plan.highlight ? "featured" : ""} reveal reveal-hidden reveal-delay-${i + 1}`}
              >
                {/* Featured gradient top */}
                {plan.highlight && (
                  <div
                    className="absolute top-0 left-0 right-0 h-1"
                    style={{
                      background: "linear-gradient(90deg, #011363, #2563EB)",
                    }}
                  />
                )}

                <div className="p-7 lg:p-8">
                  {/* Badge */}
                  {plan.badge && (
                    <div className="mb-4">
                      <span
                        className="tag text-white text-[11px] font-bold"
                        style={{ background: plan.color }}
                      >
                        {plan.badge}
                      </span>
                    </div>
                  )}

                  {/* Icon + Name */}
                  <div className="flex items-center gap-4 mb-5">
                    <div
                      className="w-14 h-14 rounded-2xl flex items-center justify-center flex-shrink-0"
                      style={{ background: plan.bg }}
                    >
                      <Icon
                        name={plan.icon}
                        size={28}
                        className="text-[#011363]"
                      />
                    </div>
                    <div>
                      <h3 className="font-heading font-bold text-lg text-[#0D1340] leading-tight">
                        {plan.name}
                      </h3>
                      <p className="text-xs text-[#9ea3ba] mt-0.5">
                        {plan.tagline}
                      </p>
                    </div>
                  </div>

                  {/* Price */}
                  <div className="mb-5 pb-5 border-b border-[#E2E5F0]">
                    <div
                      className="font-heading font-extrabold text-2xl lg:text-3xl"
                      style={{ color: plan.color }}
                    >
                      {plan.price}
                    </div>
                    <div className="text-xs text-[#9ea3ba] mt-1">
                      {plan.priceNote}
                    </div>
                  </div>

                  {/* Description */}
                  <p className="text-sm text-[#6B7280] leading-relaxed mb-6">
                    {plan.description}
                  </p>

                  {/* Features */}
                  <ul className="space-y-2.5 mb-8">
                    {plan.features.map((feat) => (
                      <li
                        key={feat.text}
                        className="flex items-center gap-2.5 text-sm"
                      >
                        {feat.included ? (
                          <Icon
                            name="CheckCircleIcon"
                            size={16}
                            className="flex-shrink-0"
                            style={{ color: plan.color } as React.CSSProperties}
                          />
                        ) : (
                          <Icon
                            name="MinusCircleIcon"
                            size={16}
                            className="text-[#D1D5DB] flex-shrink-0"
                          />
                        )}
                        <span
                          className={feat.included ? "text-[#374151]" : "text-[#9CA3AF]"}
                        >
                          {feat.text}
                        </span>
                      </li>
                    ))}
                  </ul>

                  {/* CTA */}
                  <Link
                    href={plan.ctaHref}
                    className={`w-full flex items-center justify-center gap-2 py-3 px-6 rounded-xl font-heading font-semibold text-sm transition-all duration-300 hover:-translate-y-1 ${
                      plan.highlight
                        ? "btn-primary" :"btn-secondary"
                    }`}
                    style={
                      !plan.highlight
                        ? { borderColor: plan.color, color: plan.color }
                        : {}
                    }
                  >
                    {plan.cta}
                    <Icon name="ArrowRightIcon" size={16} />
                  </Link>
                </div>
              </div>
            ))}
          </div>

          {/* Custom CTA */}
          <div className="mt-12 text-center reveal reveal-hidden">
            <p className="text-[#6B7280] text-sm mb-4">
              Precisa de uma solução que combine múltiplos modelos?
            </p>
            <Link
              href="/homepage#contact"
              className="btn-primary inline-flex text-sm"
            >
              <Icon name="ChatBubbleLeftIcon" size={16} />
              <span>Solicitar Proposta Personalizada</span>
            </Link>
          </div>
        </div>
      </section>

      {/* FAQ Section */}
      <section className="py-16 bg-[#F8F9FF]">
        <div className="max-w-3xl mx-auto px-6 lg:px-8">
          <div className="text-center mb-12 reveal reveal-hidden">
            <div className="section-label inline-flex mb-4">
              <Icon name="QuestionMarkCircleIcon" size={14} />
              Perguntas Frequentes
            </div>
            <h2 className="font-heading text-4xl font-bold text-[#0D1340] tracking-tight mt-4">
              Dúvidas sobre{" "}
              <span className="gradient-text">contratação</span>
            </h2>
          </div>

          <div className="space-y-3">
            {faqs.map((faq, i) => (
              <div
                key={i}
                className={`bg-white rounded-2xl border border-[#E2E5F0] overflow-hidden transition-all duration-300 hover:border-[#011363]/20 hover:shadow-card reveal reveal-hidden reveal-delay-${(i % 4) + 1}`}
              >
                <button
                  className="w-full flex items-center justify-between p-5 text-left"
                  onClick={() => setOpenFaq(openFaq === i ? null : i)}
                  aria-expanded={openFaq === i}
                >
                  <div className="flex items-center gap-4">
                    <span className="text-xs font-bold font-heading text-[#011363]/40 tracking-widest w-6 flex-shrink-0">
                      {String(i + 1).padStart(2, "0")}
                    </span>
                    <span className="font-heading font-semibold text-[#0D1340] text-sm pr-4">
                      {faq.q}
                    </span>
                  </div>
                  <div
                    className={`w-8 h-8 rounded-full flex-shrink-0 flex items-center justify-center transition-all duration-300 ${
                      openFaq === i
                        ? "bg-[#011363] text-white rotate-45"
                        : "bg-[#EEF0FA] text-[#011363]"
                    }`}
                  >
                    <Icon name="PlusIcon" size={16} />
                  </div>
                </button>
                {openFaq === i && (
                  <div className="px-5 pb-5 pl-[3.75rem] animate-fade-in-up">
                    <p className="text-sm text-[#6B7280] leading-relaxed">
                      {faq.a}
                    </p>
                  </div>
                )}
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Final CTA */}
      <section
        className="py-24 relative overflow-hidden"
        style={{
          background: "linear-gradient(135deg, #010D45 0%, #011363 40%, #0A1F8F 100%)",
        }}
      >
        <div
          className="absolute top-0 right-0 w-96 h-96 rounded-full opacity-10"
          style={{
            background: "radial-gradient(circle, #2563EB 0%, transparent 70%)",
            transform: "translate(30%, -30%)",
          }}
        />
        <div
          className="absolute inset-0 opacity-5"
          style={{
            backgroundImage:
              "linear-gradient(to right, white 1px, transparent 1px), linear-gradient(to bottom, white 1px, transparent 1px)",
            backgroundSize: "48px 48px",
          }}
        />

        <div className="relative z-10 max-w-3xl mx-auto px-6 lg:px-8 text-center reveal reveal-hidden">
          <div className="w-16 h-16 rounded-2xl bg-white/10 flex items-center justify-center mx-auto mb-6">
            <Icon name="RocketLaunchIcon" size={32} className="text-white" />
          </div>
          <h2 className="font-heading text-4xl lg:text-5xl font-extrabold text-white tracking-tight mb-4">
            Pronto para construir{" "}
            <span className="text-[#9ea3ba]">seu sistema?</span>
          </h2>
          <p className="text-white/70 text-lg leading-relaxed mb-10 max-w-xl mx-auto">
            Agende uma conversa sem compromisso. Analisamos sua operação e
            apresentamos a melhor solução para o seu negócio.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <a
              href="mailto:contato@advopack.com.br"
              className="inline-flex items-center justify-center gap-2 bg-white text-[#011363] font-heading font-bold py-4 px-8 rounded-xl hover:bg-[#EEF0FA] transition-all duration-300 hover:-translate-y-1 hover:shadow-lg text-sm"
            >
              <Icon name="EnvelopeIcon" size={18} />
              Solicitar Proposta Personalizada
            </a>
            <Link
              href="/systems"
              className="inline-flex items-center justify-center gap-2 bg-white/10 text-white font-heading font-semibold py-4 px-8 rounded-xl border border-white/20 hover:bg-white/20 transition-all duration-300 hover:-translate-y-1 text-sm"
            >
              Ver Sistemas
              <Icon name="ArrowRightIcon" size={18} />
            </Link>
          </div>

          <div className="flex flex-wrap items-center justify-center gap-6 mt-10 pt-8 border-t border-white/10">
            {[
              { icon: "ClockIcon" as const, text: "Resposta em até 24h" },
              { icon: "ShieldCheckIcon" as const, text: "NDA disponível" },
              { icon: "DocumentCheckIcon" as const, text: "Proposta detalhada" },
            ].map((item) => (
              <div key={item.text} className="flex items-center gap-2 text-white/60 text-sm">
                <Icon name={item.icon} size={15} className="text-white/40" />
                {item.text}
              </div>
            ))}
          </div>
        </div>
      </section>
    </div>
  );
}